package com.example.biblioteca

import android.util.Log
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.setValue
import androidx.lifecycle.ViewModel
import androidx.lifecycle.viewModelScope
import com.google.gson.annotations.SerializedName
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.flow.asStateFlow
import kotlinx.coroutines.launch

// --- Modelos de Datos ---

data class Category(val id: Int, val name: String)

data class Book(
    val id: Int,
    val title: String,
    val author: String,
    @SerializedName("category_name") val categoryName: String?,
    val content: String
)

data class Loan(val bookTitle: String, val userName: String, val dueDate: String)

data class User(
    val id: Int,
    val name: String,
    val email: String
)

data class LoginRequest(val email: String, val password: String)

data class RegisterRequest(val name: String, val email: String, val password: String)

data class AuthResponse(
    val success: Boolean,
    val message: String,
    val user: User? = null
)

// --- Estados de la UI ---

sealed interface AuthUiState {
    object Idle : AuthUiState
    object Loading : AuthUiState
    data class Success(val response: AuthResponse) : AuthUiState
    data class Error(val message: String) : AuthUiState
}

// --- ViewModels ---

class AuthViewModel : ViewModel() {
    private val _uiState = MutableStateFlow<AuthUiState>(AuthUiState.Idle)
    val uiState: StateFlow<AuthUiState> = _uiState.asStateFlow()

    fun login(email: String, pass: String) {
        viewModelScope.launch {
            _uiState.value = AuthUiState.Loading
            try {
                val response = Api.retrofitService.login(LoginRequest(email, pass))
                _uiState.value = AuthUiState.Success(response)
            } catch (e: Exception) {
                // ¡NUEVO! Imprime el error detallado en Logcat
                Log.e("AuthViewModel", "Login failed: ${e.message}", e)
                _uiState.value = AuthUiState.Error(e.message ?: "Error desconocido")
            }
        }
    }

    fun register(name: String, email: String, pass: String) {
        viewModelScope.launch {
            _uiState.value = AuthUiState.Loading
            try {
                val response = Api.retrofitService.register(RegisterRequest(name, email, pass))
                _uiState.value = AuthUiState.Success(response)
            } catch (e: Exception) {
                // ¡NUEVO! Imprime el error detallado en Logcat
                Log.e("AuthViewModel", "Registration failed: ${e.message}", e)
                _uiState.value = AuthUiState.Error(e.message ?: "Error desconocido")
            }
        }
    }

    fun resetState() {
        _uiState.value = AuthUiState.Idle
    }
}

class CategoryViewModel : ViewModel() {
    var categories by mutableStateOf<List<Category>>(emptyList())
        private set
    var isLoading by mutableStateOf(true)
        private set

    init { fetchCategories() }

    private fun fetchCategories() {
        viewModelScope.launch {
            try {
                categories = Api.retrofitService.getCategories()
            } catch (e: Exception) { 
                Log.e("CategoryViewModel", "Fetch categories failed: ${e.message}", e)
            }
            finally { isLoading = false }
        }
    }
}

class BookViewModel : ViewModel() {
    var books by mutableStateOf<List<Book>>(emptyList())
        private set
    var isLoading by mutableStateOf(true)
        private set

    init { fetchBooks() }

    private fun fetchBooks() {
        viewModelScope.launch {
            try {
                books = Api.retrofitService.getBooks()
            } catch (e: Exception) { 
                Log.e("BookViewModel", "Fetch books failed: ${e.message}", e)
            }
            finally { isLoading = false }
        }
    }

    fun getBookById(bookId: Int?): Book? {
        return books.find { it.id == bookId }
    }

    fun getBooksByCategory(categoryName: String?): List<Book> {
        return books.filter { it.categoryName == categoryName }
    }
}
