package com.example.biblioteca

import android.os.Bundle
import android.widget.Toast
import androidx.activity.ComponentActivity
import androidx.activity.compose.setContent
import androidx.activity.enableEdgeToEdge
import androidx.compose.foundation.layout.*
import androidx.compose.foundation.lazy.LazyColumn
import androidx.compose.foundation.lazy.items
import androidx.compose.material3.*
import androidx.compose.runtime.*
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.text.input.PasswordVisualTransformation
import androidx.compose.ui.unit.dp
import androidx.lifecycle.viewmodel.compose.viewModel
import androidx.navigation.NavController
import androidx.navigation.compose.NavHost
import androidx.navigation.compose.composable
import androidx.navigation.compose.rememberNavController
import com.example.biblioteca.ui.theme.BibliotecaTheme

class MainActivity : ComponentActivity() {
    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        enableEdgeToEdge()
        setContent {
            BibliotecaTheme {
                Surface(modifier = Modifier.fillMaxSize()) {
                    AppNavigation()
                }
            }
        }
    }
}

// --- Navigation ---
@Composable
fun AppNavigation() {
    val navController = rememberNavController()
    // Los ViewModels se obtienen directamente, ya que están en el mismo paquete.
    val bookViewModel: BookViewModel = viewModel()
    val authViewModel: AuthViewModel = viewModel()

    NavHost(navController = navController, startDestination = "login") {
        composable("login") { LoginScreen(navController, authViewModel) }
        composable("welcome") { WelcomeScreen(navController) }
        composable("category") { CategoryScreen(navController, viewModel()) }
        composable("books") { BooksScreen(navController, bookViewModel) }
        composable("loans") { LoansScreen(navController) }
        composable("users") { UsersScreen(navController) }
        composable("category_books/{categoryName}") { backStackEntry ->
            val categoryName = backStackEntry.arguments?.getString("categoryName")
            CategoryBooksScreen(navController = navController, categoryName = categoryName, bookViewModel = bookViewModel)
        }
        composable("book_details/{bookId}") { backStackEntry ->
            val bookId = backStackEntry.arguments?.getString("bookId")?.toIntOrNull()
            BookDetailsScreen(navController = navController, bookId = bookId, bookViewModel = bookViewModel)
        }
    }
}

// --- Screens ---
@Composable
fun LoginScreen(navController: NavController, authViewModel: AuthViewModel) {
    var name by remember { mutableStateOf("") }
    var email by remember { mutableStateOf("") }
    var password by remember { mutableStateOf("") }
    var isRegisterMode by remember { mutableStateOf(false) }

    val context = LocalContext.current
    val authState by authViewModel.uiState.collectAsState()

    LaunchedEffect(authState) {
        when (val state = authState) {
            is AuthUiState.Success -> {
                Toast.makeText(context, state.response.message, Toast.LENGTH_LONG).show()
                if (state.response.success) {
                    navController.navigate("welcome") { popUpTo("login") { inclusive = true } }
                }
                authViewModel.resetState()
            }
            is AuthUiState.Error -> {
                Toast.makeText(context, state.message, Toast.LENGTH_LONG).show()
                authViewModel.resetState()
            }
            else -> Unit
        }
    }

    Column(
        modifier = Modifier.fillMaxSize().padding(32.dp),
        horizontalAlignment = Alignment.CenterHorizontally,
        verticalArrangement = Arrangement.Center
    ) {
        val isLoading = authState is AuthUiState.Loading

        Text(if (isRegisterMode) "Registro" else "Iniciar Sesión", style = MaterialTheme.typography.headlineLarge)
        Spacer(modifier = Modifier.height(32.dp))

        if (isRegisterMode) {
            OutlinedTextField(value = name, onValueChange = { name = it }, label = { Text("Nombre Completo") }, modifier = Modifier.fillMaxWidth(), singleLine = true, enabled = !isLoading)
            Spacer(Modifier.height(16.dp))
        }

        OutlinedTextField(value = email, onValueChange = { email = it }, label = { Text("E-mail") }, modifier = Modifier.fillMaxWidth(), singleLine = true, enabled = !isLoading)
        Spacer(Modifier.height(16.dp))

        OutlinedTextField(value = password, onValueChange = { password = it }, label = { Text("Contraseña") }, visualTransformation = PasswordVisualTransformation(), modifier = Modifier.fillMaxWidth(), singleLine = true, enabled = !isLoading)
        Spacer(Modifier.height(24.dp))

        if (isLoading) {
            CircularProgressIndicator()
        } else {
            Button(
                onClick = {
                    if (isRegisterMode) {
                        authViewModel.register(name, email, password)
                    } else {
                        authViewModel.login(email, password)
                    }
                },
                modifier = Modifier.fillMaxWidth(),
                enabled = !isLoading
            ) {
                Text(if (isRegisterMode) "Crear Cuenta" else "Entrar")
            }
        }

        Spacer(Modifier.height(16.dp))
        TextButton(onClick = { isRegisterMode = !isRegisterMode }, enabled = !isLoading) {
            Text(if (isRegisterMode) "¿Ya tienes una cuenta? Inicia Sesión" else "¿No tienes una cuenta? Regístrate")
        }
    }
}


@Composable
fun WelcomeScreen(navController: NavController) {
    Column(modifier = Modifier.fillMaxSize(), verticalArrangement = Arrangement.Center, horizontalAlignment = Alignment.CenterHorizontally) {
        Text("¡Bienvenido!", style = MaterialTheme.typography.headlineLarge)
        Spacer(modifier = Modifier.height(32.dp))
        Button(onClick = { navController.navigate("category") }, modifier = Modifier.width(200.dp)) { Text("Categoría") }
        Spacer(modifier = Modifier.height(16.dp))
        Button(onClick = { navController.navigate("books") }, modifier = Modifier.width(200.dp)) { Text("Libros") }
        Spacer(modifier = Modifier.height(16.dp))
        Button(onClick = { navController.navigate("loans") }, modifier = Modifier.width(200.dp)) { Text("Préstamos") }
        Spacer(modifier = Modifier.height(16.dp))
        Button(onClick = { navController.navigate("users") }, modifier = Modifier.width(200.dp)) { Text("Usuarios") }
        Spacer(modifier = Modifier.height(48.dp))
        Button(onClick = { navController.navigate("login") { popUpTo("login") { inclusive = true } } }) { Text("Cerrar Sesión") }
    }
}

@OptIn(ExperimentalMaterial3Api::class)
@Composable
fun CategoryScreen(navController: NavController, viewModel: CategoryViewModel) {
    Column(modifier = Modifier.fillMaxSize().padding(16.dp), horizontalAlignment = Alignment.CenterHorizontally) {
        Text("Categorías", style = MaterialTheme.typography.headlineMedium)
        Spacer(modifier = Modifier.height(16.dp))

        if (viewModel.isLoading) {
            CircularProgressIndicator()
        } else {
            LazyColumn(modifier = Modifier.weight(1f).fillMaxWidth()) {
                items(viewModel.categories) { category ->
                    Card(
                        onClick = { navController.navigate("category_books/${category.name}") },
                        modifier = Modifier.fillMaxWidth().padding(vertical = 4.dp)
                    ) {
                        Text(text = category.name, modifier = Modifier.padding(16.dp))
                    }
                }
            }
        }

        Spacer(modifier = Modifier.height(16.dp))
        Button(onClick = { navController.popBackStack() }) { Text("Volver") }
    }
}

@OptIn(ExperimentalMaterial3Api::class)
@Composable
fun CategoryBooksScreen(navController: NavController, categoryName: String?, bookViewModel: BookViewModel) {
    val filteredBooks = bookViewModel.getBooksByCategory(categoryName)

    Column(modifier = Modifier.fillMaxSize().padding(16.dp), horizontalAlignment = Alignment.CenterHorizontally) {
        Text("Libros de ${categoryName ?: ""}", style = MaterialTheme.typography.headlineMedium)
        Spacer(modifier = Modifier.height(16.dp))

        if (bookViewModel.isLoading) {
            CircularProgressIndicator()
        } else if (filteredBooks.isNotEmpty()) {
            LazyColumn(modifier = Modifier.weight(1f).fillMaxWidth()) {
                items(filteredBooks) { book ->
                    Card(
                        onClick = { navController.navigate("book_details/${book.id}") },
                        modifier = Modifier.fillMaxWidth().padding(vertical = 4.dp)
                    ) {
                        Column(modifier = Modifier.padding(16.dp)) {
                            Text(book.title, style = MaterialTheme.typography.titleMedium)
                            Text(book.author, style = MaterialTheme.typography.bodySmall)
                        }
                    }
                }
            }
        } else {
            Box(modifier = Modifier.weight(1f), contentAlignment = Alignment.Center) {
                Text("No hay libros en esta categoría.")
            }
        }

        Spacer(modifier = Modifier.height(16.dp))
        Button(onClick = { navController.popBackStack() }) { Text("Volver") }
    }
}

@Composable
fun BookDetailsScreen(navController: NavController, bookId: Int?, bookViewModel: BookViewModel) {
    val book = bookViewModel.getBookById(bookId)

    Column(modifier = Modifier.fillMaxSize().padding(16.dp), horizontalAlignment = Alignment.CenterHorizontally) {
        if (bookViewModel.isLoading) {
            CircularProgressIndicator()
        } else if (book != null) {
            Text(book.title, style = MaterialTheme.typography.headlineMedium)
            Spacer(modifier = Modifier.height(8.dp))
            Text(book.author, style = MaterialTheme.typography.titleMedium)
            Spacer(modifier = Modifier.height(16.dp))
            Text(book.content)
            Spacer(modifier = Modifier.weight(1f))
            Button(onClick = { navController.popBackStack() }) { Text("Volver") }
        } else {
            Text("Libro no encontrado")
            Spacer(modifier = Modifier.height(16.dp))
            Button(onClick = { navController.popBackStack() }) { Text("Volver") }
        }
    }
}

@OptIn(ExperimentalMaterial3Api::class)
@Composable
fun BooksScreen(navController: NavController, bookViewModel: BookViewModel) {
    Column(modifier = Modifier.fillMaxSize().padding(16.dp), horizontalAlignment = Alignment.CenterHorizontally) {
        Text("Todos los Libros", style = MaterialTheme.typography.headlineMedium)
        Spacer(modifier = Modifier.height(16.dp))

        if (bookViewModel.isLoading) {
            CircularProgressIndicator()
        } else {
            LazyColumn(modifier = Modifier.weight(1f).fillMaxWidth()) {
                items(bookViewModel.books) { book ->
                    Card(
                        onClick = { navController.navigate("book_details/${book.id}") },
                        modifier = Modifier.fillMaxWidth().padding(vertical = 4.dp)
                    ) {
                        Column(modifier = Modifier.padding(16.dp)) {
                            Text(book.title, style = MaterialTheme.typography.titleMedium)
                            Text(book.author, style = MaterialTheme.typography.bodySmall)
                        }
                    }
                }
            }
        }

        Spacer(modifier = Modifier.height(16.dp))
        Button(onClick = { navController.popBackStack() }) { Text("Volver") }
    }
}

@Composable
fun LoansScreen(navController: NavController) {
    val loans = listOf(
        Loan("Cien años de soledad", "Ana Torres", "20/10/2024"),
        Loan("1984", "Carlos Ruiz", "25/10/2024")
    )
    Column(modifier = Modifier.fillMaxSize().padding(16.dp), horizontalAlignment = Alignment.CenterHorizontally) {
        Text("Préstamos", style = MaterialTheme.typography.headlineMedium)
        Spacer(modifier = Modifier.height(16.dp))
        LazyColumn(modifier = Modifier.weight(1f).fillMaxWidth()) {
            items(loans) { loan ->
                Card(modifier = Modifier.fillMaxWidth().padding(vertical = 4.dp)) {
                    Column(modifier = Modifier.padding(16.dp)) {
                        Text("Libro: ${loan.bookTitle}", style = MaterialTheme.typography.titleMedium)
                        Text("Usuario: ${loan.userName}", style = MaterialTheme.typography.bodySmall)
                        Text("Fecha de entrega: ${loan.dueDate}", style = MaterialTheme.typography.bodySmall)
                    }
                }
            }
        }
        Spacer(modifier = Modifier.height(16.dp))
        Button(onClick = { navController.popBackStack() }) { Text("Volver") }
    }
}

@Composable
fun UsersScreen(navController: NavController) {
    val users = listOf(
        User(1, "Ana Torres", "ana.t@example.com"),
        User(2, "Carlos Ruiz", "carlos.r@example.com")
    )
    Column(modifier = Modifier.fillMaxSize().padding(16.dp), horizontalAlignment = Alignment.CenterHorizontally) {
        Text("Usuarios", style = MaterialTheme.typography.headlineMedium)
        Spacer(modifier = Modifier.height(16.dp))
        LazyColumn(modifier = Modifier.weight(1f).fillMaxWidth()) {
            items(users) { user ->
                Card(modifier = Modifier.fillMaxWidth().padding(vertical = 4.dp)) {
                    Column(modifier = Modifier.padding(16.dp)) {
                        Text(user.name, style = MaterialTheme.typography.titleMedium)
                        Text(user.email, style = MaterialTheme.typography.bodySmall)
                    }
                }
            }
        }
        Spacer(modifier = Modifier.height(16.dp))
        Button(onClick = { navController.popBackStack() }) { Text("Volver") }
    }
}
