package com.example.biblioteca.ui.screens

import android.widget.Toast
import androidx.compose.foundation.layout.*
import androidx.compose.foundation.lazy.LazyColumn
import androidx.compose.foundation.lazy.items
import androidx.compose.material3.*
import androidx.compose.runtime.*
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.text.input.PasswordVisualTransformation
import androidx.compose.ui.unit.dp
import androidx.navigation.NavController
import com.example.biblioteca.AuthUiState
import com.example.biblioteca.AuthViewModel
import com.example.biblioteca.BookViewModel
import com.example.biblioteca.CategoryViewModel
import com.example.biblioteca.Loan
import com.example.biblioteca.User

@Composable
fun LoginScreen(navController: NavController, authViewModel: AuthViewModel) {
    var name by remember { mutableStateOf("") }
    var email by remember { mutableStateOf("") }
    var password by remember { mutableStateOf("") }
    var isRegisterMode by remember { mutableStateOf(false) }

    val context = LocalContext.current
    val authState by authViewModel.uiState.collectAsState()

    LaunchedEffect(authState) {
        when (val state = authState) {
            is AuthUiState.Success -> {
                Toast.makeText(context, state.response.message, Toast.LENGTH_LONG).show()
                if (state.response.success) {
                    navController.navigate("welcome") { popUpTo("login") { inclusive = true } }
                }
                authViewModel.resetState()
            }
            is AuthUiState.Error -> {
                Toast.makeText(context, state.message, Toast.LENGTH_LONG).show()
                authViewModel.resetState()
            }
            else -> Unit
        }
    }

    Column(
        modifier = Modifier.fillMaxSize().padding(32.dp),
        horizontalAlignment = Alignment.CenterHorizontally,
        verticalArrangement = Arrangement.Center
    ) {
        val isLoading = authState is AuthUiState.Loading

        Text(if (isRegisterMode) "Registro" else "Iniciar Sesión", style = MaterialTheme.typography.headlineLarge)
        Spacer(modifier = Modifier.height(32.dp))

        if (isRegisterMode) {
            OutlinedTextField(value = name, onValueChange = { name = it }, label = { Text("Nombre Completo") }, modifier = Modifier.fillMaxWidth(), singleLine = true, enabled = !isLoading)
            Spacer(Modifier.height(16.dp))
        }

        OutlinedTextField(value = email, onValueChange = { email = it }, label = { Text("E-mail") }, modifier = Modifier.fillMaxWidth(), singleLine = true, enabled = !isLoading)
        Spacer(Modifier.height(16.dp))

        OutlinedTextField(value = password, onValueChange = { password = it }, label = { Text("Contraseña") }, visualTransformation = PasswordVisualTransformation(), modifier = Modifier.fillMaxWidth(), singleLine = true, enabled = !isLoading)
        Spacer(Modifier.height(24.dp))

        if (isLoading) {
            CircularProgressIndicator()
        } else {
            Button(
                onClick = {
                    if (isRegisterMode) {
                        authViewModel.register(name, email, password)
                    } else {
                        authViewModel.login(email, password)
                    }
                },
                modifier = Modifier.fillMaxWidth(),
                enabled = !isLoading
            ) {
                Text(if (isRegisterMode) "Crear Cuenta" else "Entrar")
            }
        }

        Spacer(Modifier.height(16.dp))
        TextButton(onClick = { isRegisterMode = !isRegisterMode }, enabled = !isLoading) {
            Text(if (isRegisterMode) "¿Ya tienes una cuenta? Inicia Sesión" else "¿No tienes una cuenta? Regístrate")
        }
    }
}


@Composable
fun WelcomeScreen(navController: NavController) {
    Column(modifier = Modifier.fillMaxSize(), verticalArrangement = Arrangement.Center, horizontalAlignment = Alignment.CenterHorizontally) {
        Text("¡Bienvenido!", style = MaterialTheme.typography.headlineLarge)
        Spacer(modifier = Modifier.height(32.dp))
        Button(onClick = { navController.navigate("category") }, modifier = Modifier.width(200.dp)) { Text("Categoría") }
        Spacer(modifier = Modifier.height(16.dp))
        Button(onClick = { navController.navigate("books") }, modifier = Modifier.width(200.dp)) { Text("Libros") }
        Spacer(modifier = Modifier.height(16.dp))
        Button(onClick = { navController.navigate("loans") }, modifier = Modifier.width(200.dp)) { Text("Préstamos") }
        Spacer(modifier = Modifier.height(16.dp))
        Button(onClick = { navController.navigate("users") }, modifier = Modifier.width(200.dp)) { Text("Usuarios") }
        Spacer(modifier = Modifier.height(48.dp))
        Button(onClick = { navController.navigate("login") { popUpTo("login") { inclusive = true } } }) { Text("Cerrar Sesión") }
    }
}

@OptIn(ExperimentalMaterial3Api::class)
@Composable
fun CategoryScreen(navController: NavController, viewModel: CategoryViewModel) {
    Column(modifier = Modifier.fillMaxSize().padding(16.dp), horizontalAlignment = Alignment.CenterHorizontally) {
        Text("Categorías", style = MaterialTheme.typography.headlineMedium)
        Spacer(modifier = Modifier.height(16.dp))

        if (viewModel.isLoading) {
            Box(modifier = Modifier.fillMaxSize(), contentAlignment = Alignment.Center) {
                CircularProgressIndicator()
            }
        } else if (viewModel.categories.isEmpty()) {
            Box(modifier = Modifier.fillMaxSize(), contentAlignment = Alignment.Center) {
                Text("No se encontraron categorías.")
            }
        }
        else {
            LazyColumn(modifier = Modifier.weight(1f).fillMaxWidth()) {
                items(viewModel.categories) { category ->
                    Card(
                        onClick = { navController.navigate("category_books/${category.name}") },
                        modifier = Modifier.fillMaxWidth().padding(vertical = 4.dp)
                    ) {
                        Text(text = category.name, modifier = Modifier.padding(16.dp))
                    }
                }
            }
        }

        Spacer(modifier = Modifier.height(16.dp))
        Button(onClick = { navController.popBackStack() }) { Text("Volver") }
    }
}

@OptIn(ExperimentalMaterial3Api::class)
@Composable
fun CategoryBooksScreen(navController: NavController, categoryName: String?, bookViewModel: BookViewModel) {
    val filteredBooks = bookViewModel.getBooksByCategory(categoryName)

    Column(modifier = Modifier.fillMaxSize().padding(16.dp), horizontalAlignment = Alignment.CenterHorizontally) {
        Text("Libros de ${categoryName ?: ""}", style = MaterialTheme.typography.headlineMedium)
        Spacer(modifier = Modifier.height(16.dp))

        if (bookViewModel.isLoading) {
            Box(modifier = Modifier.fillMaxSize(), contentAlignment = Alignment.Center) {
                CircularProgressIndicator()
            }
        } else if (filteredBooks.isNotEmpty()) {
            LazyColumn(modifier = Modifier.weight(1f).fillMaxWidth()) {
                items(filteredBooks) { book ->
                    Card(
                        onClick = { navController.navigate("book_details/${book.id}") },
                        modifier = Modifier.fillMaxWidth().padding(vertical = 4.dp)
                    ) {
                        Column(modifier = Modifier.padding(16.dp)) {
                            Text(book.title, style = MaterialTheme.typography.titleMedium)
                            Text(book.author, style = MaterialTheme.typography.bodySmall)
                        }
                    }
                }
            }
        } else {
            Box(modifier = Modifier.weight(1f).fillMaxWidth(), contentAlignment = Alignment.Center) {
                Text("No hay libros en esta categoría.")
            }
        }

        Spacer(modifier = Modifier.height(16.dp))
        Button(onClick = { navController.popBackStack() }) { Text("Volver") }
    }
}

@Composable
fun BookDetailsScreen(navController: NavController, bookId: Int?, bookViewModel: BookViewModel) {
    val book = bookViewModel.getBookById(bookId)

    Column(modifier = Modifier.fillMaxSize().padding(16.dp), horizontalAlignment = Alignment.CenterHorizontally) {
        if (bookViewModel.isLoading) {
            Box(modifier = Modifier.fillMaxSize(), contentAlignment = Alignment.Center) {
                CircularProgressIndicator()
            }
        } else if (book != null) {
            Text(book.title, style = MaterialTheme.typography.headlineMedium)
            Spacer(modifier = Modifier.height(8.dp))
            Text(book.author, style = MaterialTheme.typography.titleMedium)
            Spacer(modifier = Modifier.height(16.dp))
            Text(book.content)
            Spacer(modifier = Modifier.weight(1f))
            Button(onClick = { navController.popBackStack() }) { Text("Volver") }
        } else {
            Box(modifier = Modifier.fillMaxSize(), contentAlignment = Alignment.Center) {
                Text("Libro no encontrado")
            }
        }
    }
}

@OptIn(ExperimentalMaterial3Api::class)
@Composable
fun BooksScreen(navController: NavController, bookViewModel: BookViewModel) {
    Column(modifier = Modifier.fillMaxSize().padding(16.dp), horizontalAlignment = Alignment.CenterHorizontally) {
        Text("Todos los Libros", style = MaterialTheme.typography.headlineMedium)
        Spacer(modifier = Modifier.height(16.dp))

        if (bookViewModel.isLoading) {
            Box(modifier = Modifier.fillMaxSize(), contentAlignment = Alignment.Center) {
                CircularProgressIndicator()
            }
        } else if (bookViewModel.books.isEmpty()) {
            Box(modifier = Modifier.fillMaxSize(), contentAlignment = Alignment.Center) {
                Text("No se encontraron libros.")
            }
        } else {
            LazyColumn(modifier = Modifier.weight(1f).fillMaxWidth()) {
                items(bookViewModel.books) { book ->
                    Card(
                        onClick = { navController.navigate("book_details/${book.id}") },
                        modifier = Modifier.fillMaxWidth().padding(vertical = 4.dp)
                    ) {
                        Column(modifier = Modifier.padding(16.dp)) {
                            Text(book.title, style = MaterialTheme.typography.titleMedium)
                            Text(book.author, style = MaterialTheme.typography.bodySmall)
                        }
                    }
                }
            }
        }

        Spacer(modifier = Modifier.height(16.dp))
        Button(onClick = { navController.popBackStack() }) { Text("Volver") }
    }
}

@Composable
fun LoansScreen(navController: NavController) {
    // Datos de ejemplo para la pantalla de préstamos
    val loans = listOf(
        Loan("Cien años de soledad", "Ana Torres", "20/10/2024"),
        Loan("1984", "Carlos Ruiz", "25/10/2024"),
        Loan("Dune", "Elena García", "05/11/2024"),
        Loan("Sapiens: De animales a dioses", "Pedro Martínez", "12/11/2024"),
        Loan("Don Quijote de la Mancha", "Laura Fernández", "30/11/2024")
    )
    Column(modifier = Modifier.fillMaxSize().padding(16.dp), horizontalAlignment = Alignment.CenterHorizontally) {
        Text("Préstamos Activos", style = MaterialTheme.typography.headlineMedium)
        Spacer(modifier = Modifier.height(16.dp))
        LazyColumn(modifier = Modifier.weight(1f).fillMaxWidth()) {
            items(loans) { loan ->
                Card(modifier = Modifier.fillMaxWidth().padding(vertical = 4.dp)) {
                    Column(modifier = Modifier.padding(16.dp)) {
                        Text("Libro: ${loan.bookTitle}", style = MaterialTheme.typography.titleMedium)
                        Text("Usuario: ${loan.userName}", style = MaterialTheme.typography.bodySmall)
                        Text("Fecha de entrega: ${loan.dueDate}", style = MaterialTheme.typography.bodySmall)
                    }
                }
            }
        }
        Spacer(modifier = Modifier.height(16.dp))
        Button(onClick = { navController.popBackStack() }) { Text("Volver") }
    }
}

@Composable
fun UsersScreen(navController: NavController) {
    // Datos de ejemplo para la pantalla de usuarios
    val users = listOf(
        User(1, "Ana Torres", "ana.t@example.com"),
        User(2, "Carlos Ruiz", "carlos.r@example.com"),
        User(3, "Elena García", "elena.g@example.com"),
        User(4, "Pedro Martínez", "pedro.m@example.com"),
        User(5, "Laura Fernández", "laura.f@example.com"),
        User(6, "Javier López", "javier.l@example.com"),
        User(7, "Sofía González", "sofia.g@example.com")
    )
    Column(modifier = Modifier.fillMaxSize().padding(16.dp), horizontalAlignment = Alignment.CenterHorizontally) {
        Text("Usuarios Registrados", style = MaterialTheme.typography.headlineMedium)
        Spacer(modifier = Modifier.height(16.dp))
        LazyColumn(modifier = Modifier.weight(1f).fillMaxWidth()) {
            items(users) { user ->
                Card(modifier = Modifier.fillMaxWidth().padding(vertical = 4.dp)) {
                    Column(modifier = Modifier.padding(16.dp)) {
                        Text(user.name, style = MaterialTheme.typography.titleMedium)
                        Text(user.email, style = MaterialTheme.typography.bodySmall)
                    }
                }
            }
        }
        Spacer(modifier = Modifier.height(16.dp))
        Button(onClick = { navController.popBackStack() }) { Text("Volver") }
    }
}
